/*
 *  Copyright (c) 2008 Luca Abeni
 *
 *  This is free software; see GPL.txt
 */
#include <sys/time.h>
#include <time.h>
#include <string.h>

#include <avformat.h>

#include "pktqueue.h"
#include "engine.h"
#include "dbg.h"

struct queuable_packet {
    AVPacket pkt;
    AVFormatContext *s;
    struct queuable_packet *next;
    int64_t time;
};

static int (*notifier)(int index, void *c);
static struct event_handler *h;

static void really_send_packet(AVFormatContext *s, AVPacket *p)
{
    AVFrame avframe; //FIXME/XXX remove this
    AVCodecContext *oc;
    int stream;

    oc = s->streams[0]->codec;
    avcodec_get_frame_defaults(&avframe);
    oc->coded_frame= &avframe;
    avframe.key_frame = p->flags & PKT_FLAG_KEY;
    stream = p->stream_index;
    p->stream_index = 0;

    dbg_print("Write Frame[%p]: %Ld\n", p, p->dts);
    av_interleaved_write_frame(s, p);

    av_free_packet(p);
    
    if (notifier) {
        notifier(stream, s->streams[0]->priv_data);
    }
}

static int send_handler(uint64_t t, void *c)
{
    struct queuable_packet *p;

    p = c;

    dbg_print("Sending Packet scheduled at time %Lu / %Lu\n", t, p->time);
    dbg_level_inc();
    really_send_packet(p->s, &p->pkt);
    
    av_free(p);
    dbg_level_dec();

    return 0;
}

static int packet_queue(AVFormatContext *s, struct AVPacket *pkt, int64_t t)
{
    struct queuable_packet *pkt1;

    pkt1 = av_malloc(sizeof(struct queuable_packet));
    memcpy(&pkt1->pkt, pkt, sizeof(AVPacket));
    pkt1->pkt.data = av_malloc(pkt->size);
    memcpy(pkt1->pkt.data, pkt->data, pkt->size);
    pkt1->s = s;
    pkt1->time = t;

    timed_event_new(h, t, send_handler, pkt1);
    av_free_packet(pkt);

    return 0;
}

int process_frame(AVFormatContext *s, struct AVPacket *pkt, int64_t send_time)
{
    struct timeval tv;

    gettimeofday(&tv, NULL);
    if (send_time < tv.tv_sec * 1000000LL + tv.tv_usec) {
	
        dbg_print("Sending Frame: %Ld (%Ld) --- %d %p\n",
                  send_time, pkt->dts, s->streams[0]->pts_wrap_bits,
                  s->streams[0]->priv_data);
        dbg_level_inc();
        really_send_packet(s, pkt);
        dbg_level_dec();

	return -1;
    } else {
        dbg_print("Queueing Frame: %Ld (%Ld) --- %p\n",
                  send_time, pkt->dts, s->streams[0]->priv_data);
        packet_queue(s, pkt, send_time);

	return pkt->stream_index;
    }
}

void pktqueue_init(int (*send_h)(int index, void *c), void *events_h)
{
    notifier = send_h;
    h = events_h;
}
