/*
 *  Copyright (c) 2008 Luca Abeni
 *
 *  This is free software; see GPL.txt
 */
#include <stdint.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "engine.h"

struct event {
    int fd;
    int (*handler)(int fd, void *c);
    void *context;
    struct event *next;
};

struct timed_event {
    int64_t time;
    int (*handler)(uint64_t time, void *c);
    void *context;
    struct timed_event *next;
};

struct event_handler {
    struct event *first;
    struct timed_event *first_timed_event;
};

static void list_print(struct event *p)
{
    while(p) {
        fprintf(stderr, "\t%p %d\n", p, p->fd);
        p = p->next;
    }
}

static uint64_t time_get(void)
{
    int res;
    struct timeval tv;
    int64_t t;

    res = gettimeofday(&tv, NULL);
    if (res < 0) {
        return 0;
    }
    t = tv.tv_sec * 1000000LL + tv.tv_usec;

    return t;
}

static int time_to_fire(int64_t t)
{
    int64_t now;

    now = time_get();

    return (t < now);
}

static int64_t next_event_time(struct event_handler *h)
{
    int64_t tout;

    if (h->first_timed_event == NULL) {
        return -1;
    }

    tout = time_get();
    if (tout == 0) {
        return 0;
    }
    tout = h->first_timed_event->time - tout;
    if (tout < 0) {
        tout = 0;
    }

    return tout;
}

static void timed_events_handle(struct event_handler *h)
{
    struct timed_event *p;

    if (h->first_timed_event == NULL) {
        return;
    }

    while (time_to_fire(h->first_timed_event->time)) {
        p = h->first_timed_event;
        h->first_timed_event = p->next;
        p->handler(p->time, p->context);
        free(p);
        if (h->first_timed_event == NULL) {
            return;
        }
    }
}

static int mask_prepare(struct event *first, fd_set *set)
{
    struct event *p;
    int max = 0;
    
    FD_ZERO(set);

    p = first;
    while (p != NULL) {
        FD_SET(p->fd, set);
        if (p->fd > max) {
            max = p->fd;
        }
	p = p->next;
    }

    return max;
}

static void events_cleanup(struct event_handler *h)
{
    struct event *p, *p1;

    while ((h->first) && (h->first->fd == -1)) {
        p = h->first;
	h->first = p->next;
	free(p);
    }

    if (h->first) {
        p = h->first;
        while (p->next) {
            if (p->next->fd == -1) {
                p1 = p->next;
                p->next = p1->next;
                free(p1);
            } else {
                p = p->next;
            }
	}
    }
}

/* Handle all connections */
static void mask_handle(struct event *first, fd_set *set)
{
    struct event *p;

    p = first;
    while (p != NULL) {
        if ((p->fd >= 0) && (FD_ISSET(p->fd, set))) {
            p->handler(p->fd, p->context);
        }
	p = p->next;
    }
}

void *timed_event_new(struct event_handler *h, int64_t time,int (*handler)(uint64_t time, void *c), void *context)
{
    struct timed_event *c, *p, *p1;

    /* Negative time: relative interval */
    if (time < 0) {
        time = time_get() - time;
    }

    c = malloc(sizeof(struct timed_event));
    if (c == NULL) {
        return NULL;
    }

    c->time = time;
    c->handler = handler;
    c->context = context;
    c->next = NULL;

    p = h->first_timed_event;
    p1 = NULL;
    while (p && (c->time > p->time)) {
      p1 = p;
      p = p->next;
    }

    if (p1 == NULL) {
        c->next = h->first_timed_event;
        h->first_timed_event = c;
    } else {
        c->next = p1->next;
        p1->next = c;
    }

    return c;
}

int event_loop(struct event_handler *h)
{
    fd_set in_set;
    int res, max;

    int64_t tout;
    struct timeval tv;

    tout = next_event_time(h);
    max = mask_prepare(h->first, &in_set);

    if (tout < 0){
        if (max == 0) {
            return 0;
        }
        res = select(max + 1, &in_set, NULL, NULL, NULL);
    }
    else{
        tv.tv_sec = tout / 1000000LL;
        tv.tv_usec = tout % 1000000LL;
        res = select(max+1, &in_set, NULL, NULL, &tv);
    }
    if (res < 0) {
        perror("Select");

        return -1;
    }

    timed_events_handle(h);
    mask_handle(h->first, &in_set);
    events_cleanup(h);

    return 1;
}

int event_new(struct event_handler *h, int fd, int (*handler)(int fd, void *c), void *context)
{
    struct event *c;

    c = malloc(sizeof(struct event));
    if (c == NULL) {
        perror("MAlloc");

        return -1;
    }

    c->fd = fd;
    c->handler = handler;
    c->context = context;
    c->next = h->first;
    h->first = c;

    return 1;
}

int event_del(struct event_handler *h, int fd)
{
    struct event *p;

    p = h->first;
    while (p) {
        if (p->fd == fd) {
            p->fd = -1;
            return 1;
        }
        p = p->next;
    }
    return 0;
}

int timed_event_del(struct event_handler *h, void *h1)
{
    struct timed_event *p;
    struct timed_event *event = h1;

    if (event == NULL) {
	return 0;
    }

    if (h->first_timed_event == event) {
	h->first_timed_event = event->next;
	free(event);
	return 1;
    }

    p = h->first_timed_event;
    while (p) {
        if (p->next == event) {
            p->next = event->next;
	    free(event);
            return 1;
        }
        p = p->next;
    }

    return -1;
}

struct event_handler *event_handler_init(void)
{
    struct event_handler *h;

    h = malloc(sizeof(struct event_handler));
    if (h == NULL) {
        perror("MAlloc");

        return NULL;
    }
    memset(h, 0, sizeof(struct event_handler));

    return h;
}

void event_handler_cleanup(struct event_handler *h)
{
    while (h->first) {
        struct event *p;

	p = h->first;
	h->first = p->next;
	free(p);
    }

    free(h);
}

