/*
 *  Copyright (c) 2008 Luca Abeni
 *
 *  This is free software; see GPL.txt
 */
#include <stdio.h>
#include <string.h>

#include <avformat.h>

#include "pktqueue.h"
#include "dbg.h"

#define TTL 5

struct rtp_context {
    AVFormatContext *in_s;
    AVFormatContext *out_s[MAX_STREAMS];
    int queued_frames[MAX_STREAMS];
    int64_t first_time;
};

static int multicast_address(const char *addr)
{
    return !strncmp(addr, "224", 3);	/* FIXME! */
}

static AVFormatContext *open_input_stream(const char *fname)
{
    AVFormatContext *s;
    int res;

    res = av_open_input_file(&s, fname, NULL, 0, NULL);
    if (res < 0) {
        fprintf(stderr, "Error opening %s: %d\n", fname, res);

        return NULL;
    }
    s->flags |= AVFMT_FLAG_GENPTS;

    res = av_find_stream_info(s);
    if (res < 0) {
        fprintf(stderr, "Cannot find codec parameters for %s\n", fname);

        return NULL;
    }

    dump_format(s, 0, fname, 0);

    return s;
}

static AVFormatContext *prepare_output(const char *dst, int port, int type)
{
    AVFormatContext *s;
    static AVOutputFormat *rtp_format;
    AVStream *st;
    AVCodecContext *c;

    s = av_alloc_format_context();
    rtp_format = guess_format("rtp", NULL, NULL);
    if (!rtp_format) {
        fprintf(stderr, "Unable for find the RTP format for output\n");

        return NULL;
    }

    s->oformat = rtp_format;
    st = av_new_stream(s, 0);
    if (!st) {
        fprintf(stderr, "Cannot allocate stream\n");

        return NULL;
    }
 
    c = st->codec;
    avcodec_get_context_defaults2(c, type);

    /* FIXME: global headers stuff... */

    st->stream_copy = 1;
    c->codec_type = type;

    if (dst) {
        if (multicast_address(dst)) {
            snprintf(s->filename, sizeof(s->filename),
                     "rtp://%s:%d?multicast=1&ttl=%d", dst, port, TTL);
        } else {
            snprintf(s->filename, sizeof(s->filename), "rtp://%s:%d", dst, port);
        }
        /* open the UDP sockets for RTP and RTCP */
        if (url_fopen(&s->pb, s->filename, URL_WRONLY) < 0) {
            fprintf(stderr, "Cannot open '%s'\n", s->filename);
            /* FIXME: Free the stream! */

            return NULL;
	}
    } else {
        s->filename[0] = 0;
        url_open_dyn_packet_buf(&s->pb, 1472);
    }

    return s;
}

static void close_output(AVFormatContext *ctx)
{
    ctx->streams[0]->priv_data = NULL;
    av_write_trailer(ctx);
    if (ctx->filename[0]) {
        url_fclose(ctx->pb);
    } else {
        uint8_t *buff;

        url_close_dyn_buf(ctx->pb, &buff);
        av_free(buff);
    }
    av_free(ctx->streams[0]->codec);
    av_free(ctx->streams[0]);
    av_free(ctx);
}

static int codec_setup(AVCodecContext *c, AVStream *is)
{
    AVCodecContext *ic;

    ic = is->codec;
    c->codec_id = ic->codec_id;
    c->codec_type = ic->codec_type;
    if(!c->codec_tag) {
        c->codec_tag = ic->codec_tag;
    }
    c->bit_rate = ic->bit_rate;

    /* FIXME: ExtraData ??? */
    c->extradata= ic->extradata;
    c->extradata_size= ic->extradata_size;
    
    if(av_q2d(ic->time_base) > av_q2d(is->time_base) && av_q2d(is->time_base) < 1.0/1000) {
        c->time_base = ic->time_base;
    } else {
        c->time_base = is->time_base;
    }
    
    switch(c->codec_type) {
        case CODEC_TYPE_AUDIO:
            c->sample_rate = ic->sample_rate;
            /* c->time_base= (AVRational){1, c->sample_rate}; */
            c->channels = ic->channels;
            c->frame_size = ic->frame_size;
            c->block_align= ic->block_align;

	    break;
        case CODEC_TYPE_VIDEO:
            c->pix_fmt = ic->pix_fmt;
            c->width = ic->width;
            c->height = ic->height;
            c->has_b_frames = ic->has_b_frames;

	    break;
        default:
            fprintf(stderr, "Strange Codec Type %d\n", c->codec_type);

            return -1;
    }

    return 0;
}

static void time_base_convert(AVPacket *pkt, AVStream *inst, AVStream *ost)
{
  if (pkt->pts != AV_NOPTS_VALUE) {
    pkt->pts = av_rescale_q(pkt->pts, inst->time_base, ost->time_base);
  }
  if (pkt->dts != AV_NOPTS_VALUE) {
    pkt->dts = av_rescale_q(pkt->dts, inst->time_base, ost->time_base);
  }
  if (pkt->duration != 0) {
    pkt->duration = av_rescale_q(pkt->duration, inst->time_base, ost->time_base);
  }
}

/* Return value:
 * 	-1	---> frame sent now, must read another
 * 	-2 	---> EOF
 * 	n >= 0	---> queued packet for stream n
 */
static int read_frame(struct rtp_context *rtp_c)
{
    AVPacket pkt;
    AVStream *st;
    int res;
    int64_t send_time;

    if(rtp_c->in_s == NULL) {
        return -2;
    }

    res = av_read_frame(rtp_c->in_s, &pkt);
    if (res < 0) {
        return -2;
    }

    st = rtp_c->in_s->streams[pkt.stream_index];
    dbg_print("%d: %Ld %Ld\n", pkt.stream_index, pkt.pts, pkt.dts);
    if (pkt.dts == AV_NOPTS_VALUE) {
        fprintf(stderr, "WTF??? Unknown DTS???\n");

	return -2;
    }
    send_time = av_rescale_q(pkt.dts, st->time_base, AV_TIME_BASE_Q);
    time_base_convert(&pkt, st, rtp_c->out_s[pkt.stream_index]->streams[0]);
    if (st->start_time != AV_NOPTS_VALUE) {
        send_time -= av_rescale_q(st->start_time,
                                  st->time_base, AV_TIME_BASE_Q);
    }
    dbg_print("Read Frame: %Ld (%Ld)\n", send_time, pkt.dts);
    rtp_c->queued_frames[pkt.stream_index]++;

    send_time += rtp_c->first_time;

    return process_frame(rtp_c->out_s[pkt.stream_index], &pkt, send_time);
}

int send_frame(struct rtp_context *rtp_c, int i)
{
    while(1) {
        int res;

        dbg_level_inc();
        res = read_frame(rtp_c);
        dbg_level_dec();
        dbg_print("Read Frame returned %d\n", res);
        if (res > -2) {
            dbg_level_inc();
            dbg_print("So queued frames increased to %d\n",
                      rtp_c->queued_frames[res]);
            dbg_level_dec();
        }
	if (res == i) {
            return 1;
        }
        if (res == -2) {
            return -1;
        }
    }
}

int rtp_num_streams(struct rtp_context *rtp_c)
{
    return rtp_c->in_s->nb_streams;
}

int rtp_dequeue_frame(struct rtp_context *rtp_c, int stream)
{
    return --rtp_c->queued_frames[stream];
}

struct rtp_context *rtp_stream_create(const char *fname, const char *dst, int port[], int delay)
{
    struct rtp_context *rtp_c;
    int i, res;
    
    av_register_all();

    rtp_c = av_mallocz(sizeof(struct rtp_context));
    if (rtp_c == NULL) {
        fprintf(stderr, "Cannot allocate RTP context\n");

        return NULL;
    }
    memset(rtp_c, 0, sizeof(struct rtp_context));
    
    rtp_c->in_s = open_input_stream(fname);
    if (rtp_c->in_s == NULL) {
        fprintf(stderr, "Cannot open input stream\n");

	goto err1;
    }
    
    for (i = 0; i < rtp_c->in_s->nb_streams; i++) {
	rtp_c->out_s[i] = prepare_output(dst, port ? port[i] : 0,
                                         rtp_c->in_s->streams[i]->codec->codec_type);
        if (rtp_c->out_s[i] == NULL) {
            fprintf(stderr, "Cannot open RTP stream %d\n", i);

	    goto err2;
	}
	rtp_c->queued_frames[i] = 0;
	res = codec_setup(rtp_c->out_s[i]->streams[0]->codec, rtp_c->in_s->streams[i]);
        if (res < 0) {
            fprintf(stderr, "Cannot setup out avcodeccontext %d\n", i);

            goto err2;
        }
	av_set_parameters(rtp_c->out_s[i], NULL);
	res = av_write_header(rtp_c->out_s[i]);
        if (res < 0) {
            fprintf(stderr, "Cannot Initialize output stream %d\n", i);

            goto err2;
        }
        dump_format(rtp_c->out_s[i], i, rtp_c->out_s[i]->filename, 1);
    }
    
    rtp_c->first_time = av_gettime() + delay;
    /* rtp_c->first_time -= send_time; */

    return rtp_c;

err2:
    for (i = 0; i < rtp_c->in_s->nb_streams; i++) {
        close_output(rtp_c->out_s[i]);
    }
    av_close_input_file(rtp_c->in_s);

err1:
    av_free(rtp_c);

    return NULL;
}

void rtp_set_completion(struct rtp_context *rtp_c, int (*hnd)(int index, void *c), void *ctx, void *h)
{
    int i;

    pktqueue_init(hnd, h);
    for (i = 0; i < rtp_c->in_s->nb_streams; i++) {
	rtp_c->out_s[i]->streams[0]->priv_data = ctx;
    }
}

int rtp_get_buffer(struct rtp_context *rtp_c, int stream, uint8_t **buff)
{
    return url_close_dyn_buf(rtp_c->out_s[stream]->pb, buff);
}

void rtp_release_buffer(struct rtp_context *rtp_c, int stream, uint8_t *buff)
{

    av_free(buff);
    url_open_dyn_packet_buf(&rtp_c->out_s[stream]->pb, 1472);
}

int rtp_stream_destroy(struct rtp_context *rtp_c)
{
    int i, cnt;

    if (rtp_c == NULL) {
       return -1;
    }

    if (rtp_c->in_s) {
        av_close_input_file(rtp_c->in_s);
        rtp_c->in_s = NULL;
    }

    cnt = 0;
    for (i = 0; i < MAX_STREAMS; i++) {
        cnt += rtp_c->queued_frames[i];
        if ((rtp_c->queued_frames[i] == 0) && (rtp_c->out_s[i])){
            close_output(rtp_c->out_s[i]);
            rtp_c->out_s[i] = NULL;
        }            
    }
    if (cnt == 0) {
        av_free(rtp_c);

        return 1;
    }

    return 0;
}

int rtp_get_sdp(struct rtp_context *rtp_c, char *buff, int size)
{
    return avf_sdp_create(rtp_c->out_s, rtp_c->in_s->nb_streams, buff, size);
}
